
#include <linux/wait.h>
#include <linux/list.h>

#include "dev-share.h"
#include "iosc-internal-dev.h"

#include "iprq.h"
#include "iprq_common.h"

extern struct iosc_dev_t *iosc_dev[DEVICE_NR];

static int iprqL_send_packet_handler(int request, unsigned int *msg)
{
	unsigned int listid = msg[0];
	int memid = (int)msg[1];
	struct iosc_packet_queue_t *to = NULL;
	struct list_head *head = NULL;
	struct list_head *packet;
	spinlock_t *lock = NULL;
	unsigned long flags;
	int major = GET_MAJOR(listid);
	int minor = GET_MINOR(listid);
	int queue = GET_QUE_TYPE(listid);
	int pactype = GET_PACKET_TYPE(listid);

#ifdef IPRQ_ASSERT
	if (!iosc_dev[major]) {
		IPRQ_ABORT();
	}
#endif
	if (queue == LID_QUE_REQUEST) {
		to = &iosc_dev[major]->port[minor].request;
	} else if (queue == LID_QUE_REPLY) {
		to = &iosc_dev[major]->port[minor].reply;
	} else if (queue == LID_QUE_FIFO) {
		to = &iosc_dev[major]->port[minor].fifo;
#ifdef IPRQ_ASSERT
	} else {
		IPRQ_ABORT();
#endif
	}
	if (pactype == LID_PACKET_CTL) {
		head = &to->ctlpacket_list;
		lock = &to->ctlpacket_lock;
	} else if (pactype == LID_PACKET_DAT) {
		head = &to->datapacket_list;
		lock = &to->datapacket_lock;
#ifdef IPRQ_ASSERT
	} else {
		IPRQ_ABORT();
#endif
	}
	packet = (struct list_head*)LATTACH_MEM((int)memid);
#ifdef IPRQ_ASSERT
	if (!packet) {
		IPRQ_ABORT();
	}
#endif
	spin_lock_irqsave(lock, flags);
	list_add_tail(packet, head);
	spin_unlock_irqrestore(lock, flags);

	LDETACH_MEM(packet);
	return 0;
}

static int iprqL_wake_up_handler(int request, unsigned int *msg)
{
	unsigned int listid = (unsigned int)msg[0];
	wait_queue_head_t *wait = NULL;
	int major = GET_MAJOR(listid);
	int minor = GET_MINOR(listid);
	int queue = GET_QUE_TYPE(listid);

#ifdef IPRQ_ASSERT
	if (!iosc_dev[major]) {
		IPRQ_ABORT();
	}
#endif
	if (queue == LID_QUE_REQUEST) {
		wait = &iosc_dev[major]->port[minor].request.wait;
	} else if (queue == LID_QUE_REPLY) {
		wait = &iosc_dev[major]->port[minor].reply.wait;
	} else if (queue == LID_QUE_FIFO) {
		wait = &iosc_dev[major]->port[minor].fifo.wait;
	} else if (queue == LID_QUE_BLOCK) {
		int blk = GET_DATA(listid);
		wait = &iosc_dev[major]->port[minor].block[blk].wait;
	} else if (queue == LID_QUE_OPEN) {
		wait = &iosc_dev[major]->port[minor].open_wait;
#ifdef IPRQ_ASSERT
	} else {
		IPRQ_ABORT();
#endif
	}
	wake_up_interruptible(wait);
	return 0;
}

static int iprqL_block_request_handler(int request, unsigned int *msg)
{
	unsigned int listid = msg[0];
	int memid = (int)msg[1];
	struct iosc_data_block_t *block;
	unsigned long flags;
	int major = GET_MAJOR(listid);
	int minor = GET_MINOR(listid);
	int blk = GET_DATA(listid);
	char *shmem;

#ifdef IPRQ_ASSERT
	if (GET_QUE_TYPE(listid) != LID_QUE_BLOCK) {
		IPRQ_ABORT();
	}
	if (GET_PACKET_TYPE(listid) != LID_PACKET_DAT) {
		IPRQ_ABORT();
	}
	if (!iosc_dev[major]) {
		IPRQ_ABORT();
	}
#endif
	block = &iosc_dev[major]->port[minor].block[blk];
	spin_lock_irqsave(&block->lock, flags);
	if (block->data != NULL) {
		LFREE_MEM_ADDR(block->data - sizeof(size_t));
		block->data = NULL;
		block->length = 0;
	}
	spin_unlock_irqrestore(&block->lock, flags);
	if (memid >= 0) {
		shmem = (char *)LATTACH_MEM(memid);
#ifdef IPRQ_ASSERT
		if (!shmem) {
			IPRQ_ABORT();
		}
#endif
		spin_lock_irqsave(&block->lock, flags);
		block->data = (char *)shmem + sizeof(size_t);
		block->length = *(size_t *)shmem;
		spin_unlock_irqrestore(&block->lock, flags);
		LDETACH_MEM(shmem);
	} else {
		spin_lock_irqsave(&block->lock, flags);
		block->data = NULL;
		block->length = 0;
		spin_unlock_irqrestore(&block->lock, flags);
	}
	return 0;
}

#if 1
static int iprqL_get_state(int request, unsigned int *msg)
{
	unsigned int param = msg[0];
	unsigned int listid = msg[1];
	int ret = 0;
	int major = GET_MAJOR(listid);
	int minor = GET_MINOR(listid);

#ifdef IPRQ_ASSERT
	if (!iosc_dev[major]) {
		IPRQ_ABORT();
	}
#endif
	switch (param) {
	case STATE_OPEN_FLAG:
		ret = iosc_dev[major]->port[minor].open_flag;
		break;
	case STATE_TYPE:
		ret = iosc_dev[major]->port[minor].type;
		break;
#ifdef IPRQ_ASSERT
	default:
		IPRQ_ABORT();
		break;
#endif
	}
	return ret;
}
#endif

void entry_iprq(void)
{
	iprqL_entry(IPRQ_WAKE_UP, iprqL_wake_up_handler);
	iprqL_entry(IPRQ_SEND_PACKET, iprqL_send_packet_handler);
	iprqL_entry(IPRQ_BLOCK_REQUEST, iprqL_block_request_handler);
	iprqL_entry(IPRQ_GET_STATE, iprqL_get_state);
}
